<?php
require_once __DIR__ . '/../../app/config.php';
rp_require_login(['admin','manager']);

/**
 * Auto-planner v3
 * In:  { week_start: 'YYYY-MM-DD', departments: [ids], staffing: {deptId: {1..7: concurrentCount}}, max_shift_hours?: int, use_business_hours?: bool }
 * Out: { ok:true, shifts:[ {department_id, title, start_at, end_at, user_ids: [uid], dept_name, user_names: [..], break_minutes:int } ], warnings:[...] }
 *
 * Regels:
 * - Respecteert business_hours (open/close) per afdeling/dag als 'use_business_hours' true is; anders 09:00–18:00.
 * - Max 8 uur per dienst (configureerbaar via max_shift_hours).
 * - Concurrency = gewenste gelijktijdige bezetting. We plannen zoveel "lanes" tegelijk.
 * - Beschikbaarheid: als tabel bestaat (user_availability of availability), dan filteren we daarop; anders iedereen beschikbaar.
 * - **Max 1 shift per medewerker per dag (over alle afdelingen).**
 * - Kiest medewerkers uit de afdeling (user_departments), balanceert op al geplande uren in de week.
 */

$in = rp_json_input();
$weekStart = $in['week_start'] ?? null;
$deptIds   = is_array($in['departments'] ?? null) ? array_values(array_unique(array_map('intval', $in['departments']))) : [];
$staffing  = is_array($in['staffing'] ?? null) ? $in['staffing'] : [];
$maxHours  = (int)($in['max_shift_hours'] ?? 8);
$useBH     = (bool)($in['use_business_hours'] ?? true);

if (!$weekStart || !preg_match('/^\d{4}-\d{2}-\d{2}$/', $weekStart)) rp_json(['ok'=>false,'error'=>'week_start ontbreekt of ongeldig (YYYY-MM-DD)'], 422);
if (empty($deptIds)) rp_json(['ok'=>false,'error'=>'departments is leeg'], 422);
if ($maxHours < 4 || $maxHours > 10) $maxHours = 8; // sane range

// week-datums (maandag = weekStart)
$base = strtotime($weekStart.' 00:00:00');
$dates = [];
for ($i=0;$i<7;$i++){ $dates[$i+1] = date('Y-m-d', strtotime("+$i day", $base)); } // 1..7 => YYYY-MM-DD

try {
  // Departments + naam
  $ph = implode(',', array_fill(0, count($deptIds), '?'));
  $stmD = $pdo->prepare("SELECT id, name FROM departments WHERE id IN ($ph) AND is_active=1");
  $stmD->execute($deptIds);
  $departments = [];
  foreach ($stmD->fetchAll() as $r) $departments[(int)$r['id']] = $r;

  // Business hours per dept/dag
  $bh = []; // [deptId][weekday] => ['open'=>'HH:MM:SS','close'=>'HH:MM:SS','closed'=>bool]
  if ($useBH) {
    $stmBH = $pdo->prepare("SELECT department_id, weekday, is_closed, open_time, close_time FROM business_hours WHERE department_id IN ($ph)");
    $stmBH->execute($deptIds);
    foreach ($stmBH->fetchAll() as $r) {
      $d = (int)$r['department_id']; $w = (int)$r['weekday'];
      $bh[$d][$w] = [
        'open'   => $r['open_time'] ?: '09:00:00',
        'close'  => $r['close_time'] ?: '18:00:00',
        'closed' => (int)$r['is_closed'] === 1
      ];
    }
  }
  // Fallback: als geen BH-records: 09–18 open iedere dag
  foreach ($deptIds as $dId) {
    for ($w=1;$w<=7;$w++) {
      if (!isset($bh[$dId][$w])) $bh[$dId][$w] = ['open'=>'09:00:00','close'=>'18:00:00','closed'=>false];
    }
  }

  // Kandidaten medewerkers per afdeling
  $stmU = $pdo->prepare("SELECT DISTINCT u.id, COALESCE(u.full_name, u.email) AS name, u.email, ud.department_id
                         FROM users u
                         JOIN user_departments ud ON ud.user_id = u.id
                         WHERE ud.department_id IN ($ph) AND u.is_active=1");
  $stmU->execute($deptIds);
  $usersByDept = [];
  $userNames = [];
  foreach ($stmU->fetchAll() as $r) {
    $usersByDept[(int)$r['department_id']][] = (int)$r['id'];
    $userNames[(int)$r['id']] = $r['name'] ?? $r['email'];
  }

  // Beschikbaarheid (optioneel)
  $availability = []; // [userId][weekday] => [['start'=>'HH:MM:SS','end'=>'HH:MM:SS'], ...]  (leeg = hele dag ok)
  $availLoaded = false;
  try {
    // probeer user_availability
    $q = $pdo->query("SHOW TABLES LIKE 'user_availability'"); $hasUA = (bool)$q->fetchColumn();
    if ($hasUA) {
      $rows = $pdo->query("SELECT user_id, weekday, start_time, end_time FROM user_availability")->fetchAll();
      foreach ($rows as $r) {
        $uid=(int)$r['user_id']; $w=(int)$r['weekday'];
        $availability[$uid][$w][] = ['start'=>$r['start_time'],'end'=>$r['end_time']];
      }
      $availLoaded = true;
    } else {
      // probeer availability
      $q2 = $pdo->query("SHOW TABLES LIKE 'availability'"); $hasA = (bool)$q2->fetchColumn();
      if ($hasA) {
        $rows = $pdo->query("SELECT user_id, weekday, available_from AS start_time, available_to AS end_time FROM availability")->fetchAll();
        foreach ($rows as $r) {
          $uid=(int)$r['user_id']; $w=(int)$r['weekday'];
          $availability[$uid][$w][] = ['start'=>$r['start_time'],'end'=>$r['end_time']];
        }
        $availLoaded = true;
      }
    }
  } catch(Exception $e) {
    // negeer; we vallen terug op 'iedereen beschikbaar'
  }

  // Helpers
  $toTS = fn($date,$t) => strtotime("$date $t");
  $fmt  = fn($ts) => date('Y-m-d H:i:s', $ts);
  $durH = fn($a,$b) => max(0, ($b-$a)/3600.0);

  // Al toegewezen uren per user in deze week (balanceren)
  $hoursAssigned = []; // [uid] => uren
  foreach ($userNames as $uid=>$__) $hoursAssigned[$uid] = 0.0;

  // Conflict check binnen voorlopig resultaat
  $userDaySpans = []; // [uid][date] => [[a,b],...]
  $conflicts = function($uid,$date,$a,$b) use (&$userDaySpans){
    if (!isset($userDaySpans[$uid][$date])) return false;
    foreach ($userDaySpans[$uid][$date] as [$x,$y]) {
      if ($a < $y && $b > $x) return true; // overlap
    }
    return false;
  };
  $addSpan = function($uid,$date,$a,$b) use (&$userDaySpans){ $userDaySpans[$uid][$date][] = [$a,$b]; };

  // Check availability window (if loaded)
  $isAvailable = function($uid,$w,$startT,$endT) use ($availability){
    if (empty($availability[$uid]) || empty($availability[$uid][$w])) return true; // geen data = ok
    $needS = $startT; $needE = $endT;
    foreach ($availability[$uid][$w] as $win){
      $a = strtotime("2000-01-01 ".$win['start']); $b = strtotime("2000-01-01 ".$win['end']);
      // normalize shift to same anchor day
      $s = strtotime("2000-01-01 ".date('H:i:s',$needS));
      $e = strtotime("2000-01-01 ".date('H:i:s',$needE));
      if ($s >= $a && $e <= $b) return true;
    }
    return false;
  };

  // === Stap 5: guard "max 1 dienst per user per dag" ===
  // Houdt bij wie er per dag al een dienst heeft gekregen (over alle afdelingen/segments).
  $assignedPerDay = []; // [Y-m-d][user_id] => true

  // === Plan-algoritme ===
  $result = [];
  $warnings = [];

  foreach ($deptIds as $deptId) {
    $candidates = $usersByDept[$deptId] ?? [];
    if (empty($candidates)) { $warnings[] = "Geen medewerkers gekoppeld aan afdeling #$deptId"; continue; }

    for ($w=1; $w<=7; $w++) {
      $date = $dates[$w];
      $want = (int)($staffing[$deptId][$w] ?? 0);
      $bhr  = $bh[$deptId][$w] ?? ['open'=>'09:00:00','close'=>'18:00:00','closed'=>false];

      if ($bhr['closed']) { if ($want>0) $warnings[] = "Afdeling #$deptId is gesloten op dag $w (gevraagde bezetting: $want)"; continue; }
      if ($want <= 0) continue;

      $openTS  = $toTS($date, $bhr['open']);   // bv 10:00
      $closeTS = $toTS($date, $bhr['close']);  // bv 18:00
      if ($closeTS <= $openTS) { $warnings[] = "Openingstijden inconsistent voor afdeling #$deptId dag $w"; continue; }

      $segMax   = min(max($maxHours, 4), 10); // 4..10
      // Segmenteer de dag in stukken van max segMax (bv 8u). 10–18 één segment; 10–21 => twee segmenten.
      $segments = [];
      $cursor = $openTS;
      while ($cursor < $closeTS) {
        $end = min($cursor + $segMax*3600, $closeTS);
        $segments[] = [$cursor, $end];
        if ($end == $closeTS) break;
        $cursor = $end;
      }

      // Voor elke segment maken we 'want' parallelle shifts (concurrency)
      foreach ($segments as [$segS, $segE]) {
        for ($lane=0; $lane<$want; $lane++) {
          // Kies medewerker: beschikbaar, geen overlap die dag, nog GEEN dienst die dag, en met minste uren
          $eligible = [];
          foreach ($candidates as $uid) {
            if (!empty($assignedPerDay[$date][$uid])) continue; // ✅ al een dienst op deze dag
            if ($conflicts($uid, $date, $segS, $segE)) continue;
            if (!$isAvailable($uid, $w, $segS, $segE)) continue;
            $eligible[] = $uid;
          }
          if (empty($eligible)) {
            $warnings[] = sprintf(
              "Geen beschikbare medewerker voor %s %s–%s (afdeling %s, dag %d)",
              $date, date('H:i',$segS), date('H:i',$segE), $departments[$deptId]['name'] ?? "#$deptId", $w
            );
            continue;
          }
          // balans: sorteer op al geplande uren (minst eerst)
          usort($eligible, function($a,$b) use (&$hoursAssigned){ return ($hoursAssigned[$a] <=> $hoursAssigned[$b]); });
          $uid = $eligible[0];

          // ✅ Extra safety: nogmaals check of hij al een dienst heeft op deze dag
          $dayKey = date('Y-m-d', $segS);
          if (!empty($assignedPerDay[$dayKey][$uid])) {
            continue; // zou niet moeten gebeuren door filtering, maar safety
          }
          $assignedPerDay[$dayKey][$uid] = true;

          // Pauze-regel: <6u = 0m, 6..8u = 30m, >8u = 45m
          $mins  = max(0, intval(($segE - $segS)/60));
          $break = ($mins < 360) ? 0 : (($mins <= 480) ? 30 : 45);

          $result[] = [
            'department_id' => $deptId,
            'dept_name'     => $departments[$deptId]['name'] ?? null,
            'title'         => null,
            'start_at'      => $fmt($segS),
            'end_at'        => $fmt($segE),
            'user_ids'      => [$uid],
            'user_names'    => [ $userNames[$uid] ?? ("#".$uid) ],
            'break_minutes' => $break
          ];

          $hoursAssigned[$uid] += $durH($segS,$segE);
          $addSpan($uid, $date, $segS, $segE);
        }
      }
    }
  }

  // Korte sanity: als niets gepland
  if (empty($result)) {
    rp_json(['ok'=>false,'error'=>'Er kon geen concept worden gemaakt (controleer afdelingen, openingstijden, bezetting en beschikbaarheid).','warnings'=>$warnings], 400);
  }

  rp_json(['ok'=>true,'shifts'=>$result,'warnings'=>$warnings]);
} catch (Exception $e) {
  rp_json(['ok'=>false,'error'=>$e->getMessage() ?: 'Onbekende fout'], 500);
}
