'use strict';

/**
 * Vult de 3 bestaande dashboard-vakjes met items uit /api/announcements/my-today.php
 *
 * Mapping:
 * - "Lopende acties"/"Vandaag"  → type: special
 * - "Teamactiviteiten"          → type: activity
 * - "Meldingen"                 → type: notice (nooit datums/tijden tonen)
 *
 * Nieuw:
 * - Verbergt placeholder <p class="muted"> zodra er items zijn.
 * - Bij special/activity: als zówel start als eind om 00:00 staan → toon alleen de datums.
 */

(function () {
  const HEADING_SELECTOR = 'h1,h2,h3,h4,h5,h6';
  const CARD_SELECTOR = '.card, .panel, .tile, section, article, .box, .widget, .card-like, .card-body';

  const LABELS = {
    special:  ['lopende acties','acties','speciale acties','specials','vandaag'],
    activity: ['teamactiviteiten','teamactiviteit','activiteiten','activity'],
    notice:   ['meldingen','melding','mededelingen','notices']
  };

  const role = (document.body?.dataset?.role || '').toLowerCase();
  const norm = (t)=> (t||'').toString().trim().toLowerCase();
  const esc  = (s)=> (s||'').replace(/[&<>"]/g, m => ({'&':'&amp;','<':'&lt;','>':'&gt;'}[m]));

  function findCardFor(labels){
    const heads = Array.from(document.querySelectorAll(HEADING_SELECTOR));
    const target = heads.find(h => labels.some(lbl => norm(h.textContent).includes(norm(lbl))));
    if (!target) return null;
    const card = target.closest(CARD_SELECTOR) || target.parentElement;
    if (!card) return null;

    let list = card.querySelector('.rp-ann-list');
    if (!list){
      list = document.createElement('div');
      list.className = 'rp-ann-list';
      const content = card.querySelector('.content, .card-content, .body, .inner, .panel-body') || card;
      content.appendChild(list);
    }
    return list;
  }

  // --- Datum/tijd helpers ---
  function splitDT(s){
    if (!s) return { d:'', t:'' };
    const parts = s.trim().split(' ');
    return { d: parts[0] || '', t: (parts[1] || '').trim() };
  }
  function isMidnight(t){
    // '00:00:00' of '00:00'
    return !t || t.startsWith('00:00');
  }
  function fmtDateOnly(s){ return splitDT(s).d; }
  function fmtDateTime(s){ const {d,t} = splitDT(s); return t ? `${d} ${t.slice(0,5)}` : d; }

  function renderMeta(start_at, end_at, type){
    if (type === 'notice') return ''; // bij Meldingen nooit tijden/datums

    const hasStart = !!start_at;
    const hasEnd   = !!end_at;

    if (!hasStart && !hasEnd) return '';

    // Als beide bestaan én beide 00:00 → alleen datums
    if (hasStart && hasEnd){
      const s = splitDT(start_at), e = splitDT(end_at);
      if (isMidnight(s.t) && isMidnight(e.t)){
        return `<div class="rp-ann-meta" style="color:#94a3b8;font-size:12px;margin-top:4px">
                  ${esc(s.d)} – ${esc(e.d)}
                </div>`;
      }
    }

    // Anders normale weergave (YYYY-MM-DD HH:MM) per aanwezig veld
    const left  = hasStart ? fmtDateTime(start_at) : '';
    const right = hasEnd   ? fmtDateTime(end_at)   : '';
    const text  = left && right ? `${left} – ${right}` : (left || right);

    return text
      ? `<div class="rp-ann-meta" style="color:#94a3b8;font-size:12px;margin-top:4px">${esc(text)}</div>`
      : '';
  }

  function renderList(el, items, type){
    if (!el) return;
    const card = el.closest(CARD_SELECTOR) || el.parentElement;

    if (!items || !items.length){
      const hasPlaceholder = !!card.querySelector('p.muted');
      el.innerHTML = hasPlaceholder ? '' : '<div class="rp-ann-empty" style="color:#94a3b8">Geen items</div>';
      return;
    }

    // Items aanwezig → verberg placeholders
    card.querySelectorAll('p.muted, .rp-ann-empty').forEach(n => { n.style.display = 'none'; });

    el.innerHTML = items.slice(0,5).map(it => {
      const meta = renderMeta(it.start_at, it.end_at, type);
      return `
        <div class="rp-ann-item" style="border:1px solid #e5e7eb;border-radius:12px;padding:8px 10px;margin:6px 0;">
          <div class="rp-ann-title" style="font-weight:600">${esc(it.title||'')}</div>
          ${it.message ? `<div class="rp-ann-msg" style="color:#64748b;margin-top:2px">${esc(it.message)}</div>` : ''}
          ${meta}
        </div>`;
    }).join('');
  }

  async function loadAndFill(){
    try{
      const res = await fetch('/api/announcements/my-today.php', { credentials: 'same-origin' });
      const data = await res.json();
      if (!res.ok || !data.ok) throw new Error(data?.error || 'Kon aankondigingen niet laden');

      const boxes = {
        special:  findCardFor(LABELS.special),
        activity: findCardFor(LABELS.activity),
        notice:   findCardFor(LABELS.notice)
      };

      renderList(boxes.special,  data.items?.special  || [], 'special');
      renderList(boxes.activity, data.items?.activity || [], 'activity');
      renderList(boxes.notice,   data.items?.notice   || [], 'notice');

      // Hint voor managers als er geen meldingen zijn
      if ((role === 'manager' || role === 'admin') && boxes.notice && (!data.items?.notice || !data.items.notice.length)) {
        boxes.notice.insertAdjacentHTML('beforeend',
          `<div style="margin-top:6px">
             <a href="/app/announcements/manage.php" style="font-size:12px;color:#4f46e5;text-decoration:none">➕ Plaats een melding</a>
           </div>`);
      }
    } catch(e){
      console.warn('[announcements]', e);
    }
  }

  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', loadAndFill);
  } else {
    loadAndFill();
  }
})();
