'use strict';

// --- Helpers ---
const emailOk = (v) => /.+@.+\..+/.test(v);
const pwOk = (v) => (v || '').length >= 8;
const inviteOk = (v) => /^[A-Z0-9-]{6,}$/.test((v || '').trim());

const els = {
  headline: document.getElementById('headline'),
  subline: document.getElementById('subline'),
  inviteField: document.getElementById('inviteField'),
  invite: document.getElementById('invite'),
  email: document.getElementById('email'),
  password: document.getElementById('password'),
  togglePw: document.getElementById('togglePw'),
  forgot: document.getElementById('forgot'),
  submitBtn: document.getElementById('submitBtn'),
  switchText: document.getElementById('switchText'),
  switchView: document.getElementById('switchView'),
  notice: document.getElementById('notice'),
  noticeMsg: document.getElementById('noticeMsg'),
  noticeIcon: document.getElementById('noticeIcon'),
  form: document.getElementById('authForm')
};

let mode = 'login'; // or 'signupWithCode'

function setMode(next) {
  mode = next;
  if (mode === 'login') {
    els.headline.textContent = 'Inloggen';
    els.subline.textContent = 'Welkom terug. Log in om door te gaan.';
    els.inviteField.hidden = true;
    els.submitBtn.textContent = 'Inloggen';
    els.switchText.textContent = 'Nog geen account?';
    els.switchView.innerHTML = '<span>Maak een account met code</span><svg class="chev" viewBox="0 0 24 24" width="18" height="18" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" aria-hidden="true"><path d="M9 18l6-6-6-6"/></svg>';
    els.password.autocomplete = 'current-password';
    els.forgot.hidden = false;
  } else {
    els.headline.textContent = 'Account aanmaken';
    els.subline.textContent = 'Vul je uitnodigingscode in om een account te maken.';
    els.inviteField.hidden = false;
    els.submitBtn.textContent = 'Account aanmaken';
    els.switchText.textContent = 'Al een account?';
    els.switchView.innerHTML = '<span>Log dan hier in</span><svg class="chev" viewBox="0 0 24 24" width="18" height="18" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" aria-hidden="true"><path d="M9 18l6-6-6-6"/></svg>';
    els.password.autocomplete = 'new-password';
    els.forgot.hidden = true;
  }
  hideNotice();
}

function showNotice(type, msg) {
  els.notice.className = 'notice ' + (type === 'ok' ? 'ok' : 'err');
  els.noticeMsg.textContent = msg;
  els.notice.style.display = 'flex';
  els.noticeIcon.textContent = type === 'ok' ? '✅' : '⚠️';
}
function hideNotice(){ els.notice.style.display = 'none'; }

els.switchView.addEventListener('click', () => {
  setMode(mode === 'login' ? 'signupWithCode' : 'login');
  els.password.value = '';
  if (els.invite) els.invite.value = '';
});

els.togglePw.addEventListener('click', () => {
  const isPw = els.password.type === 'password';
  els.password.type = isPw ? 'text' : 'password';
  els.togglePw.setAttribute('aria-label', isPw ? 'Verberg wachtwoord' : 'Toon wachtwoord');
});

els.forgot.addEventListener('click', async () => {
  if (!emailOk(els.email.value)) return showNotice('err', 'Vul een geldig e-mailadres in om te resetten.');
  await fetch('/api/auth/request-reset.php', { method:'POST', headers:{ 'Content-Type':'application/json' }, body: JSON.stringify({ email: els.email.value }) });
  showNotice('ok', 'Als het e-mailadres bestaat is er een reset-link geregistreerd.');
});

els.form.addEventListener('submit', async (e) => {
  e.preventDefault(); hideNotice();
  const email = els.email.value.trim();
  const password = els.password.value;
  const inviteCode = (els.invite && els.invite.value || '').toUpperCase().trim();

  if (!emailOk(email) || !pwOk(password)) {
    return showNotice('err', 'Controleer je e-mailadres en wachtwoord (min. 8 tekens).');
  }
  if (mode === 'signupWithCode' && !inviteOk(inviteCode)) {
    return showNotice('err', 'Vul een geldige uitnodigingscode in.');
  }

  els.submitBtn.disabled = true;
  try {
    const url = mode === 'login' ? '/api/auth/login.php' : '/api/auth/signup-with-code.php';
    const res = await fetch(url, {
      method:'POST',
      headers:{ 'Content-Type':'application/json' },
      body: JSON.stringify({ email, password, inviteCode })
    });

    const ct = res.headers.get('content-type') || '';
    if (!ct.includes('application/json')) {
      const text = await res.text();
      throw new Error('Server gaf geen JSON terug. Eerste regels:\n' + text.slice(0, 200));
    }
    const data = await res.json();
    if (!res.ok || data.ok === false) throw new Error(data?.error || 'Mislukt');

    // ✅ Succes-flow
    if (mode === 'login') {
      showNotice('ok', 'Succesvol ingelogd. Je wordt doorgestuurd…');
      // Flag voor dashboard-toast
      try { sessionStorage.setItem('rp_welcome', JSON.stringify({ msg: 'Welkom terug!', ts: Date.now() })); } catch(_){}
      setTimeout(() => { window.location.href = '/app/dashboard/'; }, 900);
    } else {
      // Nieuw account: sessie staat nu server-side, profiel is nog leeg → onboarding
      showNotice('ok', 'Account aangemaakt. Even je profiel invullen…');
      setTimeout(() => { window.location.href = '/app/onboarding/'; }, 900);
    }
  } catch (err) {
    showNotice('err', err.message || (mode === 'login' ? 'Inloggen mislukt.' : 'Account aanmaken mislukt.'));
  } finally {
    els.submitBtn.disabled = false;
  }
});

// Init
setMode('login');
